# Multi-Tenant POS Backend Setup Guide

This guide will help you set up and configure the complete NestJS backend for your multi-tenant POS system.

## Project Structure

```
backend/
├── src/
│   ├── auth/
│   │   ├── decorators/
│   │   │   ├── current-user.decorator.ts
│   │   │   └── public.decorator.ts
│   │   └── guards/
│   │       ├── jwt-auth.guard.ts
│   │       └── tenant.guard.ts
│   ├── common/
│   │   └── prisma/
│   │       └── prisma.service.ts
│   └── sales/
│       ├── dto/
│       │   ├── create-sale.dto.ts
│       │   └── sale-response.dto.ts
│       ├── sales.controller.ts
│       ├── sales.module.ts
│       └── sales.service.ts
├── prisma/
│   └── schema.prisma
├── database-schema.sql
├── SALES_DATA_FLOW.md
└── BACKEND_SETUP.md (this file)
```

## Prerequisites

1. **Node.js** (v18 or higher)
2. **PostgreSQL** (v13 or higher)
3. **npm** or **yarn**

## Installation Steps

### 1. Install Dependencies

```bash
cd backend

# Install core NestJS dependencies
npm install @nestjs/core @nestjs/common @nestjs/platform-express
npm install @nestjs/config @nestjs/jwt @nestjs/passport
npm install @nestjs/swagger

# Install Prisma
npm install prisma @prisma/client
npm install -D prisma

# Install validation and transformation
npm install class-validator class-transformer

# Install authentication
npm install passport passport-jwt bcrypt
npm install -D @types/passport-jwt @types/bcrypt

# Install other utilities
npm install rxjs reflect-metadata

# Development dependencies
npm install -D @types/node typescript ts-node nodemon
```

### 2. Environment Configuration

Create a `.env` file in the backend root:

```bash
# Database
DATABASE_URL="postgresql://username:password@localhost:5432/pos_db?schema=public"

# JWT
JWT_SECRET="your-super-secret-jwt-key-here"
JWT_EXPIRES_IN="7d"

# Application
NODE_ENV="development"
PORT="3001"

# Logging
LOG_LEVEL="debug"

# Stripe (for production)
STRIPE_SECRET_KEY="sk_test_..."
STRIPE_WEBHOOK_SECRET="whsec_..."
```

### 3. Database Setup

#### Create Database

```sql
CREATE DATABASE pos_db;
CREATE USER pos_user WITH PASSWORD 'your_password';
GRANT ALL PRIVILEGES ON DATABASE pos_db TO pos_user;
```

#### Run Schema Creation

```bash
# Run the database schema SQL file
psql -U pos_user -d pos_db -f database-schema.sql

# Or connect to PostgreSQL and run the SQL commands directly
```

#### Initialize Prisma

```bash
# Generate Prisma client
npx prisma generate

# Push schema to database (alternative to SQL file)
npx prisma db push

# (Optional) Seed database with sample data
npx prisma db seed
```

### 4. Core Application Setup

Create the main application files:

#### src/app.module.ts

```typescript
import { Module, ValidationPipe } from '@nestjs/common';
import { ConfigModule } from '@nestjs/config';
import { APP_GUARD, APP_PIPE } from '@nestjs/core';
import { ThrottlerModule, ThrottlerGuard } from '@nestjs/throttler';

// Modules
import { SalesModule } from './sales/sales.module';
import { AuthModule } from './auth/auth.module';
import { TenantModule } from './tenant/tenant.module';
import { ProductModule } from './product/product.module';

// Guards
import { JwtAuthGuard } from './auth/guards/jwt-auth.guard';

@Module({
  imports: [
    ConfigModule.forRoot({
      isGlobal: true,
    }),
    ThrottlerModule.forRoot({
      ttl: 60,
      limit: 100,
    }),
    SalesModule,
    AuthModule,
    TenantModule,
    ProductModule,
  ],
  providers: [
    {
      provide: APP_GUARD,
      useClass: JwtAuthGuard,
    },
    {
      provide: APP_GUARD,
      useClass: ThrottlerGuard,
    },
    {
      provide: APP_PIPE,
      useClass: ValidationPipe,
    },
  ],
})
export class AppModule {}
```

#### src/main.ts

```typescript
import { NestFactory } from '@nestjs/core';
import { ValidationPipe } from '@nestjs/common';
import { SwaggerModule, DocumentBuilder } from '@nestjs/swagger';
import { AppModule } from './app.module';

async function bootstrap() {
  const app = await NestFactory.create(AppModule);

  // Global validation pipe
  app.useGlobalPipes(
    new ValidationPipe({
      transform: true,
      whitelist: true,
      forbidNonWhitelisted: true,
    }),
  );

  // CORS configuration
  app.enableCors({
    origin: process.env.FRONTEND_URL || 'http://localhost:3000',
    credentials: true,
  });

  // Swagger documentation
  const config = new DocumentBuilder()
    .setTitle('Multi-Tenant POS API')
    .setDescription('Complete API for multi-tenant Point of Sale system')
    .setVersion('1.0')
    .addBearerAuth()
    .build();
  const document = SwaggerModule.createDocument(app, config);
  SwaggerModule.setup('api/docs', app, document);

  const port = process.env.PORT || 3001;
  await app.listen(port);
  console.log(`Application is running on: ${await app.getUrl()}`);
}
bootstrap();
```

### 5. JWT Strategy Setup

#### src/auth/strategies/jwt.strategy.ts

```typescript
import { Injectable, UnauthorizedException } from '@nestjs/common';
import { PassportStrategy } from '@nestjs/passport';
import { ExtractJwt, Strategy } from 'passport-jwt';
import { ConfigService } from '@nestjs/config';
import { PrismaService } from '../../common/prisma/prisma.service';

@Injectable()
export class JwtStrategy extends PassportStrategy(Strategy) {
  constructor(
    private configService: ConfigService,
    private prisma: PrismaService,
  ) {
    super({
      jwtFromRequest: ExtractJwt.fromAuthHeaderAsBearerToken(),
      ignoreExpiration: false,
      secretOrKey: configService.get('JWT_SECRET'),
    });
  }

  async validate(payload: any) {
    const user = await this.prisma.user.findUnique({
      where: { id: payload.sub },
      include: { tenant: true },
    });

    if (!user || !user.isActive) {
      throw new UnauthorizedException();
    }

    return {
      id: user.id,
      email: user.email,
      tenantId: user.tenantId,
      role: user.role,
      isActive: user.isActive,
    };
  }
}
```

### 6. Package.json Scripts

Add these scripts to your `package.json`:

```json
{
  "scripts": {
    "build": "nest build",
    "format": "prettier --write \"src/**/*.ts\" \"test/**/*.ts\"",
    "start": "nest start",
    "start:dev": "nest start --watch",
    "start:debug": "nest start --debug --watch",
    "start:prod": "node dist/main",
    "lint": "eslint \"{src,apps,libs,test}/**/*.ts\" --fix",
    "test": "jest",
    "test:watch": "jest --watch",
    "test:cov": "jest --coverage",
    "test:debug": "node --inspect-brk -r tsconfig-paths/register -r ts-node/register node_modules/.bin/jest --runInBand",
    "test:e2e": "jest --config ./test/jest-e2e.json",
    "prisma:generate": "prisma generate",
    "prisma:push": "prisma db push",
    "prisma:migrate": "prisma migrate dev",
    "prisma:studio": "prisma studio"
  }
}
```

## API Endpoints

### Sales Endpoints

```
POST   /sales              - Create a new sale
GET    /sales              - Get all sales (with pagination and filters)
GET    /sales/summary      - Get sales summary
GET    /sales/:id          - Get sale by ID
GET    /sales/by-number/:saleNumber - Get sale by sale number
GET    /sales/daily/:date  - Get sales for specific date
```

### Example API Usage

#### Create Sale

```bash
curl -X POST http://localhost:3001/sales \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer YOUR_JWT_TOKEN" \
  -d '{
    "items": [
      {
        "productId": "product-uuid-here",
        "quantity": 2
      }
    ],
    "paymentMethod": "CASH",
    "customerName": "John Doe",
    "taxAmount": 5.00,
    "discountAmount": 2.50
  }'
```

#### Get Sales

```bash
curl -X GET "http://localhost:3001/sales?page=1&limit=20&status=COMPLETED" \
  -H "Authorization: Bearer YOUR_JWT_TOKEN"
```

## Security Features

### Multi-Tenant Security

1. **Row-Level Security**: Database-level tenant isolation
2. **JWT-Based Authentication**: Secure token-based authentication
3. **Tenant Guards**: Application-level tenant access control
4. **Input Validation**: Comprehensive request validation
5. **Rate Limiting**: API rate limiting to prevent abuse

### Data Protection

1. **Password Hashing**: bcrypt with salt rounds
2. **JWT Expiration**: Configurable token expiration
3. **CORS Protection**: Configurable CORS policies
4. **SQL Injection Prevention**: Prisma ORM protection

## Performance Optimizations

### Database

1. **Indexes**: Comprehensive indexing strategy
2. **Connection Pooling**: Prisma connection pooling
3. **Query Optimization**: Efficient query patterns
4. **Transactions**: ACID compliance for critical operations

### Application

1. **Validation Pipes**: Efficient request validation
2. **Compression**: Response compression
3. **Caching**: Redis caching for frequently accessed data
4. **Pagination**: Efficient data pagination

## Monitoring & Logging

### Logging Setup

```typescript
// src/common/logger/logger.service.ts
import { Injectable, LoggerService } from '@nestjs/common';
import { ConfigService } from '@nestjs/config';

@Injectable()
export class CustomLoggerService implements LoggerService {
  constructor(private configService: ConfigService) {}

  log(message: string, context?: string) {
    // Implementation for structured logging
  }

  error(message: string, trace?: string, context?: string) {
    // Error logging with stack traces
  }

  warn(message: string, context?: string) {
    // Warning logs
  }

  debug(message: string, context?: string) {
    // Debug logs (development only)
  }
}
```

## Testing

### Unit Tests

```bash
# Run unit tests
npm run test

# Run with coverage
npm run test:cov
```

### E2E Tests

```bash
# Run end-to-end tests
npm run test:e2e
```

## Deployment

### Environment Variables

Ensure all production environment variables are set:

```bash
DATABASE_URL="postgresql://user:password@host:5432/db"
JWT_SECRET="your-production-secret"
NODE_ENV="production"
```

### Build and Start

```bash
# Build the application
npm run build

# Start in production mode
npm run start:prod
```

## Troubleshooting

### Common Issues

1. **Database Connection**: Verify DATABASE_URL and database accessibility
2. **JWT Issues**: Check JWT_SECRET configuration
3. **CORS Errors**: Configure CORS for your frontend domain
4. **Validation Errors**: Check DTO validation rules

### Debugging

```bash
# Start in debug mode
npm run start:debug

# Check database connection
npx prisma studio

# View logs
tail -f logs/application.log
```

## Next Steps

1. **Authentication Module**: Implement complete auth system
2. **Product Management**: Create product CRUD operations  
3. **Reporting Module**: Advanced analytics and reporting
4. **Stripe Integration**: Payment processing
5. **WebSocket Support**: Real-time updates
6. **File Upload**: Receipt and document handling
7. **Email Service**: Notifications and receipts
8. **Admin Panel**: Super admin management interface

This backend provides a solid foundation for your multi-tenant POS system with enterprise-grade security, scalability, and maintainability.