"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var SalesService_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SalesService = void 0;
const common_1 = require("@nestjs/common");
const prisma_service_1 = require("../common/prisma/prisma.service");
const client_1 = require("@prisma/client");
const library_1 = require("@prisma/client/runtime/library");
let SalesService = SalesService_1 = class SalesService {
    constructor(prisma) {
        this.prisma = prisma;
        this.logger = new common_1.Logger(SalesService_1.name);
    }
    async createSale(createSaleDto, user) {
        this.logger.log(`Creating sale for tenant ${user.tenantId} by user ${user.id}`);
        try {
            const tenantId = this.validateTenantAccess(user);
            this.validateSaleItems(createSaleDto.items);
            const sale = await this.prisma.$transaction(async (tx) => {
                const productsWithStock = await this.fetchAndValidateProducts(tx, createSaleDto.items, tenantId);
                const totals = this.calculateSaleTotals(productsWithStock, createSaleDto.taxAmount || 0, createSaleDto.discountAmount || 0);
                const saleNumber = await this.generateSaleNumber(tx, tenantId);
                const newSale = await this.createSaleRecord(tx, Object.assign(Object.assign({ tenantId,
                    saleNumber }, totals), { paymentMethod: createSaleDto.paymentMethod, customerName: createSaleDto.customerName, customerEmail: createSaleDto.customerEmail, customerPhone: createSaleDto.customerPhone, notes: createSaleDto.notes, processedBy: user.id }));
                const saleItems = await this.createSaleItems(tx, newSale.id, productsWithStock);
                await this.updateInventoryAndCreateAdjustments(tx, productsWithStock, newSale.id, tenantId, user.id);
                return Object.assign(Object.assign({}, newSale), { saleItems });
            }, {
                isolationLevel: client_1.Prisma.TransactionIsolationLevel.ReadCommitted,
                timeout: 10000,
            });
            this.logger.log(`Sale created successfully: ${sale.saleNumber}`);
            return this.formatSaleResponse(sale);
        }
        catch (error) {
            this.logger.error(`Failed to create sale: ${error.message}`, error.stack);
            if (error instanceof common_1.ConflictException ||
                error instanceof common_1.NotFoundException ||
                error instanceof common_1.BadRequestException) {
                throw error;
            }
            if (error instanceof client_1.Prisma.PrismaClientKnownRequestError) {
                return this.handlePrismaError(error);
            }
            throw new common_1.InternalServerErrorException('Failed to process sale');
        }
    }
    validateTenantAccess(user) {
        if (!user.tenantId) {
            throw new common_1.BadRequestException('User must belong to a tenant to create sales');
        }
        if (!user.isActive) {
            throw new common_1.BadRequestException('User account is inactive');
        }
        return user.tenantId;
    }
    validateSaleItems(items) {
        if (!items || items.length === 0) {
            throw new common_1.BadRequestException('Sale must contain at least one item');
        }
        const productIds = items.map(item => item.productId);
        const uniqueProductIds = new Set(productIds);
        if (productIds.length !== uniqueProductIds.size) {
            throw new common_1.BadRequestException('Duplicate products in sale items');
        }
    }
    async fetchAndValidateProducts(tx, items, tenantId) {
        const productIds = items.map(item => item.productId);
        const products = await tx.product.findMany({
            where: {
                id: { in: productIds },
                tenantId,
                isActive: true,
                deletedAt: null,
            },
        });
        if (products.length !== productIds.length) {
            const foundIds = products.map(p => p.id);
            const missingIds = productIds.filter(id => !foundIds.includes(id));
            throw new common_1.NotFoundException(`Products not found or inactive: ${missingIds.join(', ')}`);
        }
        return products.map(product => {
            const item = items.find(i => i.productId === product.id);
            if (product.stock < item.quantity) {
                throw new common_1.ConflictException(`Insufficient stock for product "${product.name}". Available: ${product.stock}, Requested: ${item.quantity}`);
            }
            return Object.assign(Object.assign({}, product), { requestedQuantity: item.quantity, unitPrice: item.unitPrice || product.price.toNumber() });
        });
    }
    calculateSaleTotals(products, taxAmount, discountAmount) {
        const subtotal = products.reduce((sum, product) => {
            return sum + (product.unitPrice * product.requestedQuantity);
        }, 0);
        const total = subtotal + taxAmount - discountAmount;
        if (total < 0) {
            throw new common_1.BadRequestException('Sale total cannot be negative');
        }
        return {
            subtotal,
            taxAmount,
            discountAmount,
            total,
        };
    }
    async generateSaleNumber(tx, tenantId) {
        const today = new Date();
        const datePrefix = today.toISOString().slice(0, 10).replace(/-/g, '');
        const todaySalesCount = await tx.sale.count({
            where: {
                tenantId,
                createdAt: {
                    gte: new Date(today.getFullYear(), today.getMonth(), today.getDate()),
                    lt: new Date(today.getFullYear(), today.getMonth(), today.getDate() + 1),
                },
            },
        });
        const saleNumber = `${datePrefix}-${String(todaySalesCount + 1).padStart(4, '0')}`;
        return saleNumber;
    }
    async createSaleRecord(tx, saleData) {
        return tx.sale.create({
            data: {
                tenantId: saleData.tenantId,
                saleNumber: saleData.saleNumber,
                subtotal: new library_1.Decimal(saleData.subtotal),
                taxAmount: new library_1.Decimal(saleData.taxAmount),
                discountAmount: new library_1.Decimal(saleData.discountAmount),
                total: new library_1.Decimal(saleData.total),
                paymentMethod: saleData.paymentMethod,
                customerName: saleData.customerName,
                customerEmail: saleData.customerEmail,
                customerPhone: saleData.customerPhone,
                notes: saleData.notes,
                processedBy: saleData.processedBy,
                status: client_1.SaleStatus.COMPLETED,
            },
        });
    }
    async createSaleItems(tx, saleId, products) {
        const saleItemsData = products.map(product => ({
            saleId,
            productId: product.id,
            productName: product.name,
            productSku: product.sku,
            quantity: product.requestedQuantity,
            unitPrice: new library_1.Decimal(product.unitPrice),
            totalPrice: new library_1.Decimal(product.unitPrice * product.requestedQuantity),
        }));
        await tx.saleItem.createMany({
            data: saleItemsData,
        });
        return tx.saleItem.findMany({
            where: { saleId },
        });
    }
    async updateInventoryAndCreateAdjustments(tx, products, saleId, tenantId, userId) {
        for (const product of products) {
            const previousStock = product.stock;
            const newStock = previousStock - product.requestedQuantity;
            await tx.product.update({
                where: { id: product.id },
                data: { stock: newStock },
            });
            await tx.inventoryAdjustment.create({
                data: {
                    tenantId,
                    productId: product.id,
                    adjustmentType: client_1.AdjustmentType.SALE,
                    quantityChange: -product.requestedQuantity,
                    previousStock,
                    newStock,
                    reason: 'Sale transaction',
                    referenceId: saleId,
                    adjustedBy: userId,
                },
            });
        }
    }
    formatSaleResponse(sale) {
        return {
            id: sale.id,
            tenantId: sale.tenantId,
            saleNumber: sale.saleNumber,
            customerName: sale.customerName,
            customerEmail: sale.customerEmail,
            customerPhone: sale.customerPhone,
            subtotal: sale.subtotal.toNumber(),
            taxAmount: sale.taxAmount.toNumber(),
            discountAmount: sale.discountAmount.toNumber(),
            total: sale.total.toNumber(),
            paymentMethod: sale.paymentMethod,
            status: sale.status,
            notes: sale.notes,
            processedBy: sale.processedBy,
            createdAt: sale.createdAt,
            updatedAt: sale.updatedAt,
            saleItems: sale.saleItems.map(item => ({
                id: item.id,
                productId: item.productId,
                productName: item.productName,
                productSku: item.productSku,
                quantity: item.quantity,
                unitPrice: item.unitPrice.toNumber(),
                totalPrice: item.totalPrice.toNumber(),
                createdAt: item.createdAt,
            })),
        };
    }
    handlePrismaError(error) {
        switch (error.code) {
            case 'P2002':
                throw new common_1.ConflictException('Sale number already exists');
            case 'P2025':
                throw new common_1.NotFoundException('Record not found');
            default:
                this.logger.error(`Unhandled Prisma error: ${error.code}`, error);
                throw new common_1.InternalServerErrorException('Database operation failed');
        }
    }
    async findAll(user, page = 1, limit = 20, status, startDate, endDate) {
        const tenantId = this.validateTenantAccess(user);
        const skip = (page - 1) * limit;
        const where = Object.assign(Object.assign({ tenantId }, (status && { status })), (startDate || endDate ? {
            createdAt: Object.assign(Object.assign({}, (startDate && { gte: startDate })), (endDate && { lte: endDate })),
        } : {}));
        const [sales, total] = await Promise.all([
            this.prisma.sale.findMany({
                where,
                include: { saleItems: true },
                orderBy: { createdAt: 'desc' },
                skip,
                take: limit,
            }),
            this.prisma.sale.count({ where }),
        ]);
        return {
            sales: sales.map(sale => this.formatSaleResponse(sale)),
            total,
        };
    }
    async findOne(id, user) {
        const tenantId = this.validateTenantAccess(user);
        const sale = await this.prisma.sale.findFirst({
            where: { id, tenantId },
            include: { saleItems: true },
        });
        if (!sale) {
            throw new common_1.NotFoundException('Sale not found');
        }
        return this.formatSaleResponse(sale);
    }
};
exports.SalesService = SalesService;
exports.SalesService = SalesService = SalesService_1 = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [prisma_service_1.PrismaService])
], SalesService);
//# sourceMappingURL=sales.service.js.map